package com.genexus.printing;

/** ParseINI, clase para el manejo de INIs
 * @version 1.0.1
 */

import java.util.*;
import java.io.*;

public class ParseINI
{
    private static final int MAX_LINE_LENGTH=255; // Maximo largo de una property
    private static final String GENERAL="&General&"; // Seccion General

    private String entryName;
    private DataInputStream inputStream;
    private Map<String, Object> sections = new TreeMap<String, Object>(String.CASE_INSENSITIVE_ORDER);
    private Hashtable sectionEntries;
    private Hashtable general;
    private Hashtable aliased=new Hashtable(); // Contiene las dulpas (original -> alias)
    private Hashtable alias=new Hashtable();   // Contiene las dulpas (alias -> original)
    private boolean need2Save = false;
    private String filename = null;

    byte []tempBytes=new byte[16384];

    public ParseINI()
    {
        general=new Hashtable();
        filename = null;
    }

    public ParseINI(String filename) throws IOException
    {
        this.filename = new File(filename).getAbsolutePath();
        try
        {
            FileInputStream inputStream = new FileInputStream(filename);
            load(inputStream);
            inputStream.close();
        }
        catch(FileNotFoundException fnfe)
        { // Si debo crear el archivo
            (new FileWriter(filename)).close(); // Creo el archivo
            general=new Hashtable();
        }
    }

    public boolean need2Save()
    {
        return need2Save;
    }

    /** Guardo nuevamente el .INI si este se ha modificado
     */
    public void finalize()
    {
        if(need2Save)
            try
            {
                save();
            }catch(IOException e)
            {
                System.out.println("ParseINI (" + filename + "):" +e);
            }
    }

    /** Devuelve un Enumeration con los nombres de las secciones
     */
    public Iterator sectionNames()
    {
        return sections.values().iterator();
    }	

    /** Indica si existe una seccion
     *  @param section Nombre de la secci�n
     *  @return boolean True si la secci�n existe
     */
    public boolean sectionExists(String section)
    {
        return sections.containsKey(section);
    }
	
	/** Devuelve la property asociada
	 * @param prop Nombre de la property a obtener
	 * @return el value de la property o null si no existe es property
	 */
	public String getGeneralProperty(String prop)
	{
            return getProperty(general,prop,null);
	}	

	/** Agrega una Property solo si no exist�a
	 * @param section Nombre de la Seccion
	 * @param prop Nombre de la property
	 * @param value Valor de la property
	 * @return true si se agrego la Property y false si ya exist�a
	 */
	public boolean setupProperty(String section, String prop, String value)
	{
        if(getProperty(section, prop) != null)
            return false;
        else setProperty(section, prop, value);
            return true;
	}

	/** Agrega una property a una seccion ( y crea la seccion en caso de que
	 * no exista)
	 * @param section Nombre de la seccion
	 * @param prop Nombre de la property
	 * @param value Valor de la property
	 */
	public void setProperty(String section, String prop, String value)
	{
            if(!sections.containsKey(section))
                sections.put(section, new Hashtable<String,Object>());
            need2Save |= !value.equals(((Hashtable<String,Object>)sections.get(section)).put(prop, value));
	}

	/** Devuelve la property asociada de la secci�n especificada
	 * @param section Nombre de la secci�n
	 * @param prop Nombre de la property a obtener
	 * @return el value de la property o null si no existe es property
	 */
	public String getProperty(String section, String prop)
	{
            return getProperty((Hashtable)sections.get(section), prop, null);
	}
	
	private String getProperty(Hashtable section, String prop, String defecto)
	{
            if(section != null && section.containsKey(prop))
                return (String)section.get(prop);
            return defecto;
	}	

	/** Carga de un DataInputStream las properties
	 * Las secci�n se setean con la property 'Name: SectionName' o '[SectionName]'
	 * y termina cuando aparece otro inicio de secci�n
	 * Si no se especifica ninguna seccion se toma una por defecto
	 * @param inputStream el DataInputStream
	 */
	public void load(InputStream iStream) throws IOException
	{
            inputStream=new DataInputStream(iStream);

            sectionEntries=new Hashtable();
            sections.put(GENERAL, sectionEntries);
            try
            {
                while ((entryName = readEntryName()) != null)
                    sectionEntries.put(entryName,readEntry());
            }catch(EOFException e){}
            general = (Hashtable) sections.get(GENERAL);
            sections.remove(GENERAL);
	}

	/** Guarda el INI en el archivo que se abrio este INI
	 */
	public void save() throws IOException
	{
		if(need2Save && filename != null)
			save(new FileOutputStream(filename));
	}
	
	/** Guarda el INI
	 * @param oStrem el Outputstream
	 */
	public void save(OutputStream oStream) throws IOException
	{
            DataOutputStream outputStream = new DataOutputStream(oStream);
            Enumeration props;
            Iterator secs;
            String prop, value;
            int it;
            if(general!=null)
            {
                props = general.keys();
                while(props.hasMoreElements())
                {
                    prop=(String)props.nextElement();
                    if((value = (String)getGeneralProperty(prop)).equals(""))continue;
                    if((value.length() + prop.length() + 4) > MAX_LINE_LENGTH)it = (MAX_LINE_LENGTH - prop.length() - 4);
                    else it = value.length();
                    outputStream.writeBytes(prop + "= " + value.substring(0, it) + "\r\n");
                    for(; (value.length() - it) > ( MAX_LINE_LENGTH - 4) ; it+= (MAX_LINE_LENGTH - 4))
                            outputStream.writeBytes(" " + value.substring(it, it + (MAX_LINE_LENGTH - 4)) + "\r\n");
                    if(it < value.length())outputStream.writeBytes(" " + value.substring(it) + "\r\n");
                }
            }
            secs=sectionNames();
            while(secs.hasNext())
            {
                String section=(String)secs.next();
                outputStream.writeBytes("\r\n["+section+"]\r\n");
                props = ((Hashtable)sections.get(section)).keys();
                while(props.hasMoreElements())
                {
                    prop=(String)props.nextElement();
                    if((value = (String)getProperty(section, prop)).equals(""))continue;
                    if((value.length() + prop.length() + 4) > MAX_LINE_LENGTH)it = (MAX_LINE_LENGTH - prop.length() - 4);
                    else it = value.length();
                    outputStream.writeBytes(prop + "= " + value.substring(0, it) + "\r\n");
                    for(; (value.length() - it) > ( MAX_LINE_LENGTH - 4) ; it+= MAX_LINE_LENGTH - 4)
                            outputStream.writeBytes(" " + value.substring(it, it + MAX_LINE_LENGTH - 4) + "\r\n");
                    if(it < value.length())outputStream.writeBytes(" " + value.substring(it) + "\r\n");
                }
            }
            outputStream.close();
            need2Save = false;
	}	

	/** Obtiene del InputStream el nombre del entry
	 * @param inputStream el DataInputStream
	 * @return el nombre del entry
	 */
	private String readEntryName()throws IOException
	{
            int offset=0;
            byte car;

            while(true)
            switch((char)(car=inputStream.readByte()))
            {
                case '%':  // Comentarios
                case '*':
                    while(true)
                    {
                            car=inputStream.readByte();
                            if(car=='\n' || car=='\r')break;
                    }
                    break;
                case ' ':  // Continuaci�n del entry anterior -> se concatena con el entry que ya se estaba procesando
                    if(entryName==null)throw new IOException("Invalid entry");
                    sectionEntries.put(entryName,(String) sectionEntries.get(entryName) + readEntry());
                    return readEntryName();
                case '[':  // Comienza nueva seccion
                    sectionEntries = new Hashtable();
                    String sectionName = readEntry();
                    sectionName=sectionName.substring(0,sectionName.length()-1);
                    sections.put(sectionName,sectionEntries);
                    entryName=null;
                    return readEntryName();
                case '\n':
                case '\r': break;
                default:   // Tomo entry
                    tempBytes[offset++]=car;
                    while(true)
                    {
                            car=inputStream.readByte();
                            if(/*Character.isWhitespace((char)car) ||*/
                            car==':' || car=='=')break;
                            else tempBytes[offset++]=car;
                    }
                    if(offset==4 &&         // Chequeo si es nueva Seccion
                       tempBytes[0]=='N' &&
                       tempBytes[1]=='a' &&
                       tempBytes[2]=='m' &&
                       tempBytes[3]=='e')
                    {
                            sectionEntries=new Hashtable();
                            String section=readEntry();
                            if(section.startsWith("./"))section=section.substring(2);
                            sections.put(section,sectionEntries);
                            entryName=null;
                            return readEntryName();
                    }
                    else return new String(tempBytes,0,offset);
            }
	}

	/** Obtiene del InputStream el value del entry
	 * @param inputStream el DataInputStream
	 * @return el value del entry
	 */
	private String readEntry()throws IOException
	{		
            int offset=0;
            byte car=inputStream.readByte();
            while(Character.isWhitespace((char)car))
            {
                if (car=='\0' || car=='\n' || car=='\r')
                {
                    return "";
                }
                car=inputStream.readByte();
            }
            tempBytes[offset++]=car;
            try
            {
                while((car=inputStream.readByte())!='\0' && car!='\n' && car!='\r')
                    tempBytes[offset++]=car;
            }
            catch(EOFException e){}
            return new String(tempBytes,0,offset);
	}
}
